/*----------------------------------------------------------------------------
 *      R T L  -  T C P N E T
 *----------------------------------------------------------------------------
 *      Name:    ETH_STM32X.C
 *      Purpose: Driver for ST STM32F105/107 Ethernet Controller
 *      Rev.:    V3.70
 *----------------------------------------------------------------------------
 *      This code is part of the RealView Run-Time Library.
 *      Copyright (c) 2004-2009 KEIL - An ARM Company. All rights reserved.
 *---------------------------------------------------------------------------*/

#include <Net_Config.h>
#include <stm32f10x_cl.h>
#include "ETH_STM32x.h"

/* The following macro definitions may be used to select the speed
   of the physical link:

  _10MBIT_   - connect at 10 MBit only
  _100MBIT_  - connect at 100 MBit only

  By default an autonegotiation of the link speed is used. This may take 
  longer to connect, but it works for 10MBit and 100MBit physical links.     */

/* Net_Config.c */
extern U8 own_hw_adr[];

/* Local variables */
static U8 TxBufIndex;
static U8 RxBufIndex;

/* ENET local DMA Descriptors. */
static RX_Desc Rx_Desc[NUM_RX_BUF];
static TX_Desc Tx_Desc[NUM_TX_BUF];

/* ENET local DMA buffers. */
static U32 rx_buf[NUM_RX_BUF][ETH_BUF_SIZE>>2];
static U32 tx_buf[NUM_TX_BUF][ETH_BUF_SIZE>>2];

/*----------------------------------------------------------------------------
 *      ENET Ethernet Driver Functions
 *----------------------------------------------------------------------------
 *  Required functions for Ethernet driver module:
 *  a. Polling mode: - void init_ethernet ()
 *                   - void send_frame (OS_FRAME *frame)
 *                   - void poll_ethernet (void)
 *  b. Interrupt mode: - void init_ethernet ()
 *                     - void send_frame (OS_FRAME *frame)
 *                     - void int_enable_eth ()
 *                     - void int_disable_eth ()
 *                     - interrupt function 
 *---------------------------------------------------------------------------*/

/* Local Function Prototypes */
static void rx_descr_init (void);
static void tx_descr_init (void);
static void write_PHY (U32 PhyReg, U16 Value);
static U16  read_PHY (U32 PhyReg);

/*--------------------------- init_ethernet ---------------------------------*/

void init_ethernet (void) {
   /* Initialize the ETH ethernet controller. */
   U32 regv,tout,id1,id2;

   /* Enable clock for Port A,B,C,D and Alternate function */
   RCC->APB2ENR |= 0x0000003D;

   /* Enable clock for MAC. */
   RCC->AHBENR  |= 0x0001C000;

#ifdef STM3210C
   /* Use MII on STM3210C-EVAL board. */

   /* Remap MII RX pins to PD8 .. PD12 */
   AFIO->MAPR   |= 0x00200000;

   /* Configure Port A ethernet pins. */
   GPIOA->CRL   &= 0xFFFF0000;
   GPIOA->CRL   |= 0x00004B44;
   GPIOA->CRH   &= 0xFFFFFFF0;
   GPIOA->CRH   |= 0x0000000B;

   /* Configure Port B ethernet pins */
   GPIOB->CRH   &= 0xFF0000F0;
   GPIOB->CRH   |= 0x00BBB40B;

   /* Configure Port C ethernet pins */
   GPIOC->CRL   &= 0xFFFF000F;
   GPIOC->CRL   |= 0x00004BB0;

   /* Configure Port D ethernet pins */
   GPIOD->CRH   &= 0xFFF00000;
   GPIOD->CRH   |= 0x00044444;
#else
   /* Use RMII on MCBSTM32C evaluation board. */

   /* Enable RMII, remap RX pins to PD8 .. PD10 */
   AFIO->MAPR   |= 0x00A00000;

   /* Configure Port A ethernet pins. */
   GPIOA->CRL   &= 0xFFFFF00F;
   GPIOA->CRL   |= 0x00000B40;

   /* Configure Port B ethernet pins */
   GPIOB->CRH   &= 0xFF000FFF;
   GPIOB->CRH   |= 0x00BBB000;

   /* Configure Port C ethernet pins */
   GPIOC->CRL   &= 0xFFFFFF0F;
   GPIOC->CRL   |= 0x000000B0;

   /* Configure Port D ethernet pins */
   GPIOD->CRH   &= 0xFFFFF000;
   GPIOD->CRH   |= 0x00000444;
#endif

   /* Reset Ethernet MAC */
   RCC->AHBSTR  |= 0x00004000;
   RCC->AHBSTR  &=~0x00004000;

   ETH->DMABMR  |= DBMR_SR;
   while (ETH->DMABMR & DBMR_SR);

   /* MDC Clock range 60-72MHz. */
   ETH->MACMIIAR = 0x00000000;

   /* Put the DP83848C in reset mode */
   write_PHY (PHY_REG_BMCR, 0x8000);

   /* Wait for hardware reset to end. */
   for (tout = 0; tout < 0x10000; tout++) {
      regv = read_PHY (PHY_REG_BMCR);
      if (!(regv & 0x8800)) {
         /* Reset complete, device not Power Down. */
         break;
      }
   }

   /* Check if this is a DP83848C PHY. */
   id1 = read_PHY (PHY_REG_IDR1);
   id2 = read_PHY (PHY_REG_IDR2);

   if (((id1 << 16) | (id2 & 0xFFF0)) == DP83848C_ID) {
      /* Configure the PHY device */
#if defined (_10MBIT_)
      /* Connect at 10MBit */
      write_PHY (PHY_REG_BMCR, PHY_FULLD_10M);
#elif defined (_100MBIT_)
      /* Connect at 100MBit */
      write_PHY (PHY_REG_BMCR, PHY_FULLD_100M);
#else
      /* Use autonegotiation about the link speed. */
      write_PHY (PHY_REG_BMCR, PHY_AUTO_NEG);
      /* Wait to complete Auto_Negotiation. */
      for (tout = 0; tout < 0x10000; tout++) {
         regv = read_PHY (PHY_REG_BMSR);
         if (regv & 0x0020) {
            /* Autonegotiation Complete. */
            break;
         }
      }
#endif
   }

   /* Check the link status. */
   for (tout = 0; tout < 0x10000; tout++) {
      regv = read_PHY (PHY_REG_STS);
      if (regv & 0x0001) {
         /* Link is on. */
         break;
      }
   }

   /* Initialize MAC control register */
   ETH->MACCR  = MCR_ROD;

   /* Configure Full/Half Duplex mode. */
   if (regv & 0x0004) {
      /* Full duplex is enabled. */
      ETH->MACCR |= MCR_DM;
   }

   /* Configure 100MBit/10MBit mode. */
   if ((regv & 0x0002) == 0) {
      /* 100MBit mode. */
      ETH->MACCR |= MCR_FES;
   }

   /* MAC address filtering, accept multicast packets. */
   ETH->MACFFR = MFFR_HPF | MFFR_PAM;
   ETH->MACFCR = MFCR_ZQPD;

   /* Set the Ethernet MAC Address registers */
   ETH->MACA0HR = ((U32)own_hw_adr[5] <<  8) | (U32)own_hw_adr[4];
   ETH->MACA0LR = ((U32)own_hw_adr[3] << 24) | (U32)own_hw_adr[2] << 16 |
                  ((U32)own_hw_adr[1] <<  8) | (U32)own_hw_adr[0];

   /* Initialize Tx and Rx DMA Descriptors */
   rx_descr_init ();
   tx_descr_init ();

   /* Flush FIFO, start DMA Tx and Rx */
   ETH->DMAOMR = DOMR_FTF | DOMR_ST | DOMR_SR;

   /* Enable receiver and transmiter */
   ETH->MACCR |= MCR_TE | MCR_RE;

   /* Reset all interrupts */
   ETH->DMASR  = 0xFFFFFFFF;

   /* Enable Rx and Tx interrupts. */
   ETH->DMAIER = INT_NISE | INT_RIE | INT_TIE;
}

/*--------------------------- int_enable_eth --------------------------------*/

void int_enable_eth (void) {
   /* Ethernet Interrupt Enable function. */
   NVIC->ISER[1] = 1 << 29;
}


/*--------------------------- int_disable_eth -------------------------------*/

void int_disable_eth (void) {
   /* Ethernet Interrupt Disable function. */
   NVIC->ICER[1] = 1 << 29;
}


/*--------------------------- send_frame ------------------------------------*/

void send_frame (OS_FRAME *frame) {
   /* Send frame to ETH ethernet controller */
   U32 *sp,*dp;
   U32 i,j;

   j = TxBufIndex;
   /* Wait until previous packet transmitted. */
   while (Tx_Desc[j].CtrlStat & DMA_TX_OWN);

   sp = (U32 *)&frame->data[0];
   dp = (U32 *)(Tx_Desc[j].Addr & ~3);

   /* Copy frame data to ETH IO buffer. */
   for (i = (frame->length + 3) >> 2; i; i--) {
      *dp++ = *sp++;
   }
   Tx_Desc[j].Size      = frame->length;
   Tx_Desc[j].CtrlStat |= DMA_TX_OWN;
   if (++j == NUM_TX_BUF) j = 0;
   TxBufIndex = j;
   /* Start frame transmission. */
   ETH->DMASR   = DSR_TPSS;
   ETH->DMATPDR = 0;
}


/*--------------------------- interrupt_ethernet ----------------------------*/

void ETH_IRQHandler (void) {
   /* Ethernet Controller Interrupt function. */
   OS_FRAME *frame;
   U32 i,RxLen,int_stat;
   U32 *sp,*dp;

   while (((int_stat = ETH->DMASR) & INT_NISE) != 0) {
      ETH->DMASR = int_stat;
      if (int_stat & INT_RIE) {
         /* Valid frame has been received. */
         i = RxBufIndex;
         if (Rx_Desc[i].Stat & DMA_RX_ERROR_MASK) {
            goto rel;
         }
         if ((Rx_Desc[i].Stat & DMA_RX_SEG_MASK) != DMA_RX_SEG_MASK) {
            goto rel;
         }
         RxLen = ((Rx_Desc[i].Stat >> 16) & 0x3FFF) - 4;
         if (RxLen > ETH_MTU) {
            /* Packet too big, ignore it and free buffer. */
            goto rel;
         }
         /* Flag 0x80000000 to skip sys_error() call when out of memory. */
         frame = alloc_mem (RxLen | 0x80000000);
         /* if 'alloc_mem()' has failed, ignore this packet. */
         if (frame != NULL) {
            sp = (U32 *)(Rx_Desc[i].Addr & ~3);
            dp = (U32 *)&frame->data[0];
            for (RxLen = (RxLen + 3) >> 2; RxLen; RxLen--) {
               *dp++ = *sp++;
            }
            put_in_queue (frame);
         }
         /* Release this frame from ETH IO buffer. */
rel:     Rx_Desc[i].Stat = DMA_RX_OWN;

         if (++i == NUM_RX_BUF) i = 0;
         RxBufIndex = i;
      }
      if (int_stat & INT_TIE) {
         /* Frame transmit completed. */
      }
   }
}


/*--------------------------- rx_descr_init ---------------------------------*/

static void rx_descr_init (void) {
   /* Initialize Receive DMA Descriptor array. */
   U32 i,next;

   RxBufIndex = 0;
   for (i = 0, next = 0; i < NUM_RX_BUF; i++) {
      if (++next == NUM_RX_BUF) next = 0;
      Rx_Desc[i].Stat = DMA_RX_OWN;
      Rx_Desc[i].Ctrl = DMA_RX_RCH | ETH_BUF_SIZE;
      Rx_Desc[i].Addr = (U32)&rx_buf[i];
      Rx_Desc[i].Next = (U32)&Rx_Desc[next];
   }
   ETH->DMARDLAR = (U32)&Rx_Desc[0];
}



/*--------------------------- tx_descr_init ---------------------------------*/

static void tx_descr_init (void) {
   /* Initialize Transmit DMA Descriptor array. */
   U32 i,next;

   TxBufIndex = 0;
   for (i = 0, next = 0; i < NUM_TX_BUF; i++) {
      if (++next == NUM_TX_BUF) next = 0;
      Tx_Desc[i].CtrlStat = DMA_TX_TCH | DMA_TX_LS | DMA_TX_FS;
      Tx_Desc[i].Addr     = (U32)&tx_buf[i];
      Tx_Desc[i].Next     = (U32)&Tx_Desc[next];
   }
   ETH->DMATDLAR = (U32)&Tx_Desc[0];
}


/*--------------------------- write_PHY -------------------------------------*/

static void write_PHY (U32 PhyReg, U16 Value) {
   /* Write a data 'Value' to PHY register 'PhyReg'. */
   U32 tout;

   ETH->MACMIIDR = Value;
   ETH->MACMIIAR = DP83848C_DEF_ADR << 11 | PhyReg << 6 | MMAR_MW | MMAR_MB;

   /* Wait utill operation completed */
   tout = 0;
   for (tout = 0; tout < MII_WR_TOUT; tout++) {
      if ((ETH->MACMIIAR & MMAR_MB) == 0) {
         break;
      }
   }
}


/*--------------------------- read_PHY --------------------------------------*/

static U16 read_PHY (U32 PhyReg) {
   /* Read a PHY register 'PhyReg'. */
   U32 tout;

   ETH->MACMIIAR = DP83848C_DEF_ADR << 11 | PhyReg << 6 | MMAR_MB;

  /* Wait until operation completed */
   tout = 0;
   for (tout = 0; tout < MII_RD_TOUT; tout++) {
      if ((ETH->MACMIIAR & MMAR_MB) == 0) {
         break;
      }
   }
   return (ETH->MACMIIDR & MMDR_MD);
}

/*----------------------------------------------------------------------------
 * end of file
 *---------------------------------------------------------------------------*/

