/*----------------------------------------------------------------------------
 *      R T L   T C P N E T   E x a m p l e
 *----------------------------------------------------------------------------
 *      Name:    LEDSWITCH.C
 *      Purpose: Example LED Control Server application
 *      Rev.:    V3.70
 *----------------------------------------------------------------------------
 *      This code is part of the RealView Run-Time Library.
 *      Copyright (c) 2004-2008 KEIL - An ARM Company. All rights reserved.
 *---------------------------------------------------------------------------*/

#include <RTL.h>
#include <stdio.h>
#include <stm32f10x_cl.h>
#include "GLCD.h"
#include <string.h>

U8 socket_tcp;
U8 socket_udp;

#define PORT_NUM 1001
#define BLINKLED 0x01  // Command for blink the leds on board

/*--------------------------- init ------------------------------------------*/

static void init () {
   /* Add System initialisation code here */

   /* Set the clocks. */
   SystemInit();
   RCC->APB2ENR |= 0x00000261;

   /* Configure the GPIO for LEDs. */
   GPIOD->CRL &= 0xFFF00FFF;
   GPIOD->CRL |= 0x00033000;
   GPIOD->CRH &= 0xFF0FFFFF;
   GPIOD->CRH |= 0x00300000;
   GPIOE->CRH &= 0xF0FFFFFF;
   GPIOE->CRH |= 0x03000000;

   /* Configure UART2 for 115200 baud. */
   AFIO->MAPR |= 0x00000008;
   GPIOD->CRL &= 0xF00FFFFF;
   GPIOD->CRL |= 0x04B00000;

   RCC->APB1ENR |= 0x00020000;
   USART2->BRR = 0x0135;
   USART2->CR3 = 0x0000;
   USART2->CR2 = 0x0000;
   USART2->CR1 = 0x200C;

   /* Setup and enable the SysTick timer for 100ms. */
   SysTick->LOAD = (SystemFrequency / 10) - 1;
   SysTick->CTRL = 0x05;
}


/*--------------------------- init_display ----------------------------------*/

static void init_display () {
   /* LCD Module init */

   GLCD_init();
   GLCD_clear(White);
   GLCD_setTextColor(Blue);
   GLCD_displayStringLn(Line2, "       RL-ARM");
   GLCD_displayStringLn(Line3, " LEDswitch example");
}


/*--------------------------- timer_poll ------------------------------------*/

static void timer_poll () {
   /* System tick timer running in poll mode */

   if (SysTick->CTRL & 0x10000) {
      /* Timer tick every 100 ms */
      timer_tick ();
   }
}


/*--------------------------- fputc -----------------------------------------*/

int fputc (int ch, FILE *f)  {
   /* Debug output to serial port. */

   if (ch == '\n')  {
      while (!(USART2->SR & 0x0080));
      USART2->DR = 0x0D;
   }
   while (!(USART2->SR & 0x0080));
   USART2->DR = (ch & 0xFF);
   return (ch);
}


/*--------------------------- LED_out ---------------------------------------*/

void LED_out (U32 val) {
   U32 rv;

   rv = 0;
   if (val & 0x01) rv |= 0x00004000;
   GPIOE->BSRR = rv;
   GPIOE->BRR  = rv ^ 0x00004000;

   rv = 0;
   if (val & 0x02) rv |= 0x00002000;
   if (val & 0x04) rv |= 0x00000008;
   if (val & 0x08) rv |= 0x00000010;
   GPIOD->BSRR = rv;
   GPIOD->BRR  = rv ^ 0x0002018;
}


/*--------------------------- Process received data  ------------------------*/

void procrec (U8 *buf) {
   switch (buf[0]) {
      case BLINKLED:
         LED_out (buf[1]);
         break;
   }
}
/*--------------------------- UDP socket ------------------------------------*/

U16 udp_callback (U8 soc, U8 *rip, U16 rport, U8 *buf, U16 len) {
   rip  = rip;
   rport= rport;
   len  = len;

   if (soc != socket_udp) {
      /* Check if this is the socket we are connected to */
      return (0);
   }

   procrec(buf);
   return (0);
}

/*--------------------------- TCP socket ------------------------------------*/

U16 tcp_callback (U8 soc, U8 evt, U8 *ptr, U16 par) {
   /* This function is called by the TCP module on TCP event */
   /* Check the 'Net_Config.h' for possible events.          */
   par = par;

   if (soc != socket_tcp) {
      return (0);
   }

   switch (evt) {
      case TCP_EVT_DATA:
         /* TCP data frame has arrived, data is located at *par1, */
         /* data length is par2. Allocate buffer to send reply.   */
         procrec(ptr);
         break;

      case TCP_EVT_CONREQ:
         /* Remote peer requested connect, accept it */
         return (1);

      case TCP_EVT_CONNECT:
         /* The TCP socket is connected */
         return (1);
   }
   return (0);
}


/*--------------------------- main ------------------------------------------*/

int main (void) {
   /* Main Thread of the TcpNet */

   init ();
   init_display ();
   init_TcpNet ();

   /* Initialize UDP Socket and start listening */
   socket_udp = udp_get_socket (0, UDP_OPT_SEND_CS | UDP_OPT_CHK_CS, udp_callback);
   if (socket_udp != 0) {
      udp_open (socket_udp, PORT_NUM);
   }

   /* Initialize TCP Socket and start listening */
   socket_tcp = tcp_get_socket (TCP_TYPE_SERVER, 0, 10, tcp_callback);
   if (socket_tcp != 0) {
      tcp_listen (socket_tcp, PORT_NUM);
   }

   while (1) {
      timer_poll ();
      main_TcpNet ();
   }
}


/*----------------------------------------------------------------------------
 * end of file
 *---------------------------------------------------------------------------*/


