/*----------------------------------------------------------------------------
 *      R T L   T C P N E T   E x a m p l e
 *----------------------------------------------------------------------------
 *      Name:    CLIENT.C
 *      Purpose: Example LED Control Client application
 *      Rev.:    V3.70
 *----------------------------------------------------------------------------
 *      This code is part of the RealView Run-Time Library.
 *      Copyright (c) 2004-2008 KEIL - An ARM Company. All rights reserved.
 *---------------------------------------------------------------------------*/

#include <RTL.h>
#include <stdio.h>
#include <stm32f10x_cl.h>
#include "GLCD.h"
#include <string.h>

//-------- <<< Use Configuration Wizard in Context Menu >>> -----------------

//   <h>Remote IP Address
//   ====================
//
//     <o>IP1: Address byte 1 <0-255>
//     <i> Default: 192
#define IP1            192

//     <o>IP2: Address byte 2 <0-255>
//     <i> Default: 168
#define IP2            168

//     <o>IP3: Address byte 3 <0-255>
//     <i> Default: 0
#define IP3            0

//     <o>IP4: Address byte 4 <0-255>
//     <i> Default: 100
#define IP4            100

//   </h>

//   <o>Remote Port <1-65535>
//   <i> Do not set number of port too small,
//   <i> maybe it is already used.
//   <i> Default: 1001
#define PORT_NUM       1001

//   <o>Communication Protocol <0=> TCP <1=> UDP
//   <i> Selecet a protocol for sending data.
#define PROTOCOL       1

//   <o>LED Blinking speed <1-100>
//   <i> Blinking speed = SPEED * 100ms
//   <i> Default: 2
#define SPEED          2


//------------- <<< end of configuration section >>> -----------------------


#define BLINKLED 0x01  /* Command for blink the leds on board */
#define SENDLEN  2     /* Number of bytes to send */
#define TCP      0
#define UDP      1

BOOL tick;
U8 socket_tcp;
U8 socket_udp;
U8 Rem_IP[4] = {IP1,IP2,IP3,IP4};


/*--------------------------- init ------------------------------------------*/

static void init () {
   /* Add System initialisation code here */

   /* Set the clocks. */
   SystemInit();
   RCC->APB2ENR |= 0x00000261;

   /* Configure the GPIO for LEDs. */
   GPIOD->CRL &= 0xFFF00FFF;
   GPIOD->CRL |= 0x00033000;
   GPIOD->CRH &= 0xFF0FFFFF;
   GPIOD->CRH |= 0x00300000;
   GPIOE->CRH &= 0xF0FFFFFF;
   GPIOE->CRH |= 0x03000000;

   /* Configure UART2 for 115200 baud. */
   AFIO->MAPR |= 0x00000008;
   GPIOD->CRL &= 0xF00FFFFF;
   GPIOD->CRL |= 0x04B00000;

   RCC->APB1ENR |= 0x00020000;
   USART2->BRR = 0x0135;
   USART2->CR3 = 0x0000;
   USART2->CR2 = 0x0000;
   USART2->CR1 = 0x200C;

   /* Setup and enable the SysTick timer for 100ms. */
   SysTick->LOAD = (SystemFrequency / 10) - 1;
   SysTick->CTRL = 0x05;
}


/*--------------------------- init_display ----------------------------------*/

static void init_display () {
   /* LCD Module init */

   GLCD_init();
   GLCD_clear(White);
   GLCD_setTextColor(Blue);
   GLCD_displayStringLn(Line2, "       RL-ARM");
   GLCD_displayStringLn(Line3, "   Client example");
}


/*--------------------------- LED_out ---------------------------------------*/

void LED_out (U32 val) {
   U32 rv;

   rv = 0;
   if (val & 0x01) rv |= 0x00004000;
   GPIOE->BSRR = rv;
   GPIOE->BRR  = rv ^ 0x00004000;

   rv = 0;
   if (val & 0x02) rv |= 0x00002000;
   if (val & 0x04) rv |= 0x00000008;
   if (val & 0x08) rv |= 0x00000010;
   GPIOD->BSRR = rv;
   GPIOD->BRR  = rv ^ 0x0002018;
}


/*--------------------------- timer_poll ------------------------------------*/

static void timer_poll () {
   /* System tick timer running in poll mode */

   if (SysTick->CTRL & 0x10000) {
      /* Timer tick every 100 ms */
      timer_tick ();
      tick = __TRUE;
   }
}


/*--------------------------- fputc -----------------------------------------*/

int fputc (int ch, FILE *f)  {
   /* Debug output to serial port. */

   if (ch == '\n')  {
      while (!(USART2->SR & 0x0080));
      USART2->DR = 0x0D;
   }
   while (!(USART2->SR & 0x0080));
   USART2->DR = (ch & 0xFF);
   return (ch);
}


/*--------------------------- UDP socket ------------------------------------*/

U16 udp_callback (U8 soc, U8 *rip, U16 rport, U8 *buf, U16 len) {
   /* This function is called by the UDP module when UDP packet is received. */

   /* Make a reference to suppress compiler warnings. */
   buf  = buf;
   rip  = rip;
   rport= rport;
   len  = len;
   soc  = soc;
   return (0);
}


/*--------------------------- TCP socket ------------------------------------*/

U16 tcp_callback (U8 soc, U8 evt, U8 *ptr, U16 par) {
   /* This function is called by the TCP module on TCP event */
   /* Check the Net_Config.h for possible events.            */

   /* Make a reference to suppress compiler warnings. */
   soc = soc;
   par = par;
   evt = evt;
   ptr = ptr;
   return (0);
}


/*--------------------------- TCP send --------------------------------------*/

void send_data (U8 p2val) {
   U8 *sendbuf;
   U8 p2;

   /* UDP */
   if (socket_udp != 0) {
      /* Start Connection */
      sendbuf = udp_get_buf (SENDLEN);
      sendbuf[0] = BLINKLED;
      sendbuf[1] = p2val;
      udp_send (socket_udp, Rem_IP, 1001, sendbuf, SENDLEN);
   }

   /* TCP */
   if (socket_tcp != 0) {
      /* Start Connection */
      p2 = p2val;
      switch (tcp_get_state(socket_tcp)) {
         case TCP_STATE_FREE:
         case TCP_STATE_CLOSED:
            tcp_connect (socket_tcp, Rem_IP, PORT_NUM, 0);
            break;
         case TCP_STATE_CONNECT:
            if (tcp_check_send (socket_tcp) == __TRUE) {
               sendbuf = tcp_get_buf(SENDLEN);
               sendbuf[0] = BLINKLED;
               sendbuf[1] = p2;
               tcp_send (socket_tcp, sendbuf, SENDLEN);
            }
            break;
      }
   }
}


/*--------------------------- main ------------------------------------------*/

int main (void) {
   /* Main Thread of the TcpNet */
   U8 p2val, cnt, lshf;
   U8 protocol;

   init ();
   init_display ();
   init_TcpNet ();

   protocol = PROTOCOL;
   switch (protocol) {
      case TCP:
         socket_tcp = tcp_get_socket (TCP_TYPE_CLIENT, 0, 10, tcp_callback);
         break;
      case UDP:
         socket_udp = udp_get_socket (0, UDP_OPT_SEND_CS | UDP_OPT_CHK_CS, udp_callback);
         if (socket_udp != 0) {
            udp_open (socket_udp, PORT_NUM);
         }
         break;
   }

   p2val = 1;
   cnt   = 0;
   lshf  = 1;  /* left shift */
   while (1) {
      timer_poll ();
      main_TcpNet ();
      if (tick == __TRUE) {
         if (++cnt == SPEED) {
            LED_out (p2val);
            send_data(p2val);
            p2val = lshf ? (p2val << 1) : (p2val >> 1);
            if (p2val == 0x80) lshf = 0;
            if (p2val == 0x01) lshf = 1;
            cnt = 0;
        }
        tick = __FALSE;
     }
   }
}


/*----------------------------------------------------------------------------
 * end of file
 *---------------------------------------------------------------------------*/


