/*----------------------------------------------------------------------------
 *      R T L  -  F l a s h   F i l e   S y s t e m
 *----------------------------------------------------------------------------
 *      Name:    SPI_STM32X.C 
 *      Purpose: Serial Peripheral Interface Driver for STM32x
 *      Rev.:    V3.70
 *----------------------------------------------------------------------------
 *      This code is part of the RealView Run-Time Library.
 *      Copyright (c) 2004-2008 KEIL - An ARM Company. All rights reserved.
 *---------------------------------------------------------------------------*/

#include <File_Config.h>
#include <stm32f10x_cl.h>

#ifdef STM3210C
 #define SPIx   SPI3
#else
 #define SPIx   SPI1
#endif

/* SPI_SR - bit definitions. */
#define RXNE    0x01
#define TXE     0x02
#define BSY     0x80

/*----------------------------------------------------------------------------
 *      SPI Driver Functions
 *----------------------------------------------------------------------------
 *  Required functions for SPI driver module:
 *   - void spi_init ()
 *   - void spi_ss (U32 ss)
 *   - U8   spi_send (U8 outb)
 *   - void spi_hi_speed (BOOL on)
 *---------------------------------------------------------------------------*/

/*--------------------------- spi_init --------------------------------------*/

void spi_init (void) {
   /* Initialize and enable the SSP Interface module. */

#ifdef STM3210C
   /* Use SPI3 on STM3210C-EVAL board. */

   /* Enable clock for GPIOA,C,E, AFIO and SPI3. */
   RCC->APB2ENR |= 0x00000055;
   RCC->APB1ENR |= 0x00008000;

   /* Set SPI3 remap (use PC10..PC12). */
   AFIO->MAPR   |= 0x10000000;

   /* SPI3_NSS is GPIO, output set to high. */
   GPIOA->CRL &= 0xFFF0FFFF;
   GPIOA->CRL |= 0x00030000;
   GPIOA->BSRR = 0x00000010;

   /* SPI3_SCK, SPI3_MISO, SPI3_MOSI are SPI pins. */
   GPIOC->CRH &= 0xFFF000FF;
   GPIOC->CRH |= 0x000B8B00;
#else
   /* Use SPI1 on MCBSTM32C evaluation board. */

   /* Enable clock for GPIOA,E, AFIO and SPI1. */
   RCC->APB2ENR |= 0x00001045;

   /* No SPI1 remap (use PA5..PA7). */
   AFIO->MAPR   &= 0xFFFFFFFE;

   /* SPI1_NSS is GPIO, output set to high. */
   /* SPI1_SCK, SPI1_MISO, SPI1_MOSI are SPI pins. */
   GPIOA->CRL &= 0x0000FFFF;
   GPIOA->CRL |= 0xB8B30000;
   GPIOA->BSRR = 0x00000010;
#endif

   /* Card Sensor PE.0 input */
   /* 1 = NO Card, 0 = Card plugged. */
   GPIOE->CRL &= 0xFFFFFFF0;
   GPIOE->CRL |= 0x00000004;

   /* Enable SPI in Master Mode, CPOL=0, CPHA=0. */
   /* Clock speed = fPCLK1 / 256 = 280 kHz at 72 MHz PCLK1 clk. */
   SPIx->CR1  = 0x037C;
   SPIx->CR2  = 0x0000;
}


/*--------------------------- spi_hi_speed ----------------------------------*/

void spi_hi_speed (BOOL on) {
   /* Set a SPI clock speed to desired value. */

   if (on == __TRUE) {
      /* Max. 18 MBit used for Data Transfer. */
      SPIx->CR1 &= ~0x0030;
      SPIx->CR1 |=  0x0008;
   }
   else {
      /* Max. 400 kBit used in Card Initialization. */
      SPIx->CR1 |= 0x0038;
   }
}


/*--------------------------- spi_ss ----------------------------------------*/

void spi_ss (U32 ss) {
   /* Enable/Disable SPI Chip Select (drive it high or low). */

   GPIOA->BSRR = ss ? 0x00000010 : 0x00100000;
}


/*--------------------------- spi_send --------------------------------------*/

U8 spi_send (U8 outb) {
   /* Write and Read a byte on SPI interface. */

   /* Wait if TXE cleared, Tx FIFO is full. */
   while (!(SPIx->SR & TXE));
   SPIx->DR = outb;

   /* Wait if RNE cleared, Rx FIFO is empty. */
   while (!(SPIx->SR & RXNE));
   return (SPIx->DR);
}


/*----------------------------------------------------------------------------
 * end of file
 *---------------------------------------------------------------------------*/
