/*----------------------------------------------------------------------------
 *      R T L   T C P N E T   E x a m p l e
 *----------------------------------------------------------------------------
 *      Name:    HTTP_DEMO.C
 *      Purpose: HTTP Server demo example
 *      Rev.:    V3.70
 *----------------------------------------------------------------------------
 *      This code is part of the RealView Run-Time Library.
 *      Copyright (c) 2004-2009 KEIL - An ARM Company. All rights reserved.
 *---------------------------------------------------------------------------*/

#include <stdio.h>
#include <RTL.h>
#include <Net_Config.h>
#include <stm32f10x_cl.h>
#include "GLCD.h"
#include <string.h>

BOOL LEDrun;
BOOL LCDupdate;
BOOL tick;
U32  dhcp_tout;
U8   lcd_text[2][16+1] = {" ",                /* Buffer for LCD text         */
                          "Waiting for DHCP"};

extern LOCALM localm[];                       /* Local Machine Settings      */
#define MY_IP localm[NETIF_ETH].IpAdr
#define DHCP_TOUT   50                        /* DHCP timeout 5 seconds      */

static void init_io (void);
static void init_display (void);

#ifdef RTX_KERNEL
 U64 tcp_stack[800/8];                        /* A bigger stack for tcp_task */

 /* Forward references */
 __task void init       (void);
 __task void blink_led  (void);
 __task void timer_task (void);
 __task void tcp_task   (void);
#endif

/*--------------------------- init ------------------------------------------*/

#ifndef RTX_KERNEL

static void init () {
   /* Add System initialisation code here */ 

   init_io ();
   init_display ();
   init_TcpNet ();

   /* Setup and enable the SysTick timer for 100ms. */
   SysTick->LOAD = (SystemFrequency / 10) - 1;
   SysTick->CTRL = 0x05;
}

#else

__task void init (void) {
   /* Add System initialisation code here */ 

   init_io ();
   init_display ();
   init_TcpNet ();

   /* Initialize Tasks */
   os_tsk_prio_self (100);
   os_tsk_create (blink_led, 20);
   os_tsk_create (timer_task, 30);
   os_tsk_create_user (tcp_task, 0, &tcp_stack, sizeof(tcp_stack));
   os_tsk_delete_self();
}

#endif

/*--------------------------- timer_poll ------------------------------------*/

#ifndef RTX_KERNEL

static void timer_poll () {
   /* System tick timer running in poll mode */

   if (SysTick->CTRL & 0x10000) {
      /* Timer tick every 100 ms */
      timer_tick ();
      tick = __TRUE;
   }
}

#else

__task void timer_task (void) {
   /* System tick timer task */
   os_itv_set (10);
   while (1) {
      timer_tick ();
      tick = __TRUE;
      os_itv_wait ();
   }
}

#endif

/*--------------------------- init_io ---------------------------------------*/

static void init_io () {

   /* Set the clocks. */
   SystemInit();
   RCC->APB2ENR |= 0x00000279;

   /* Configure the GPIO for Push Buttons */
   GPIOB->CRH &= 0xFFFFFF0F;
   GPIOB->CRH |= 0x00000040;
   GPIOC->CRH &= 0xFFF0FFFF;
   GPIOC->CRH |= 0x00040000;

   /* Configure the GPIO for LEDs. */
   GPIOD->CRL &= 0xFFF00FFF;
   GPIOD->CRL |= 0x00033000;
   GPIOD->CRH &= 0xFF0FFFFF;
   GPIOD->CRH |= 0x00300000;
   GPIOE->CRH &= 0xF0FFFFFF;
   GPIOE->CRH |= 0x03000000;

   /* Configure UART2 for 115200 baud. */
   AFIO->MAPR |= 0x00000008;
   GPIOD->CRL &= 0xF00FFFFF;
   GPIOD->CRL |= 0x04B00000;

   RCC->APB1ENR |= 0x00020000;
   USART2->BRR = 0x0135;
   USART2->CR3 = 0x0000;
   USART2->CR2 = 0x0000;
   USART2->CR1 = 0x200C;

   /* Configure ADC.10 input. */  //PC0
   GPIOC->CRL &= 0xFFFFFFF0;
   ADC1->SQR1  = 0x00000000;
   ADC1->SMPR1 = (1<<12);
   ADC1->SQR3  = (14<<0);
   ADC1->CR1   = 0x00000100;
   ADC1->CR2   = 0x000E0003;

   /* Reset calibration */
   ADC1->CR2  |= 0x00000008;
   while (ADC1->CR2 & 0x00000008);

   /* Start calibration */
   ADC1->CR2  |= 0x00000004;
   while (ADC1->CR2 & 0x00000004);
   ADC1->CR2  |= 0x00500000;
}


/*--------------------------- fputc -----------------------------------------*/

int fputc (int ch, FILE *f)  {
   /* Debug output to serial port. */

   if (ch == '\n')  {
      while (!(USART2->SR & 0x0080));
      USART2->DR = 0x0D;
   }
   while (!(USART2->SR & 0x0080));
   USART2->DR = (ch & 0xFF);
   return (ch);
}


/*--------------------------- LED_out ---------------------------------------*/

void LED_out (U32 val) {
   U32 rv;

   rv = 0;
   if (val & 0x01) rv |= 0x00004000;
   GPIOE->BSRR = rv;
   GPIOE->BRR  = rv ^ 0x00004000;

   rv = 0;
   if (val & 0x02) rv |= 0x00002000;
   if (val & 0x04) rv |= 0x00000008;
   if (val & 0x08) rv |= 0x00000010;
   GPIOD->BSRR = rv;
   GPIOD->BRR  = rv ^ 0x0002018;
}


/*--------------------------- AD_in -----------------------------------------*/

U16 AD_in (U32 ch) {
   /* Read ARM Analog Input */
   U32 val = 0;

   if (ch < 1) {
      val = ADC1->DR & 0x0FFF;
   }
   return (val);
}


/*--------------------------- get_button ------------------------------------*/

U8 get_button (void) {
   /* Read ARM Digital Input */
   U32 val = 0;

   if ((GPIOB->IDR & (1 << 9)) == 0) {
      /* Key button */
      val |= 0x01;
   }
   if ((GPIOC->IDR & (1 << 13)) == 0) {
      /* Wakeup button */
      val |= 0x02;
   }
   return (val);
}


/*--------------------------- upd_display -----------------------------------*/

static void upd_display () {
   /* Update LCD Module display text. */

   GLCD_clearLn (Line5);
   GLCD_displayStringLn(Line5, lcd_text[0]);
   GLCD_clearLn (Line6);
   GLCD_displayStringLn(Line6, lcd_text[1]);

   LCDupdate =__FALSE;
}


/*--------------------------- init_display ----------------------------------*/

static void init_display () {
   /* LCD Module init */

   GLCD_init();
   GLCD_clear(White);
   GLCD_setTextColor(Blue);
   GLCD_displayStringLn(Line1, "       RL-ARM");
   GLCD_displayStringLn(Line2, "    HTTP example");

   upd_display ();
}


/*--------------------------- dhcp_check ------------------------------------*/

static void dhcp_check () {
   /* Monitor DHCP IP address assignment. */

   if (tick == __FALSE || dhcp_tout == 0) {
      return;
   }
#ifdef RTX_KERNEL
   tick = __FALSE;
#endif
   if (mem_test (&MY_IP, 0, IP_ADRLEN) == __FALSE && !(dhcp_tout & 0x80000000)) {
      /* Success, DHCP has already got the IP address. */
      dhcp_tout = 0;
      sprintf((char *)lcd_text[0],"IP address:");
      sprintf((char *)lcd_text[1],"%d.%d.%d.%d", MY_IP[0], MY_IP[1],
                                                 MY_IP[2], MY_IP[3]);
      LCDupdate = __TRUE;
      return;
   }
   if (--dhcp_tout == 0) {
      /* A timeout, disable DHCP and use static IP address. */
      dhcp_disable ();
      sprintf((char *)lcd_text[1]," DHCP failed    " );
      LCDupdate = __TRUE;
      dhcp_tout = 30 | 0x80000000;
      return;
   }
   if (dhcp_tout == 0x80000000) {
      dhcp_tout = 0;
      sprintf((char *)lcd_text[0],"IP address:");
      sprintf((char *)lcd_text[1],"%d.%d.%d.%d", MY_IP[0], MY_IP[1],
                                                 MY_IP[2], MY_IP[3]);
      LCDupdate = __TRUE;
   }
}


/*--------------------------- blink_led -------------------------------------*/

#ifndef RTX_KERNEL

static void blink_led () {
   /* Blink the LEDs on an eval board */
   const U8 led_val[8] = { 0x01,0x03,0x07,0x0F,0x0E,0x0C,0x08,0x00 };
   static U32 cnt;

   if (tick == __TRUE) {
      /* Every 100 ms */
      tick = __FALSE;
      if (LEDrun == __TRUE) {
         LED_out (led_val[cnt]);
         if (++cnt >= sizeof(led_val)) {
            cnt = 0;
         }
      }
      if (LCDupdate == __TRUE) {
         upd_display ();
      }
   }
}

#else

__task void blink_led () {
   /* Blink the LEDs on an eval board */
   const U8 led_val[8] = { 0x01,0x03,0x07,0x0F,0x0E,0x0C,0x08,0x00 };
   U32 cnt = 0;

   LEDrun = __TRUE;
   while(1) {
      /* Every 100 ms */
      if (LEDrun == __TRUE) {
         LED_out (led_val[cnt]);
         if (++cnt >= sizeof(led_val)) {
            cnt = 0;
         }
      }
      if (LCDupdate == __TRUE) {
         upd_display ();
      }
      os_dly_wait(10);
   }
}

#endif


/*---------------------------------------------------------------------------*/

#ifndef RTX_KERNEL

int main (void) {
   /* Main Thread of the TcpNet */

   init ();
   LEDrun = __TRUE;
   dhcp_tout = DHCP_TOUT;
   while (1) {
      timer_poll ();
      main_TcpNet ();
      dhcp_check ();
      blink_led ();
   }
}

#else

__task void tcp_task (void) {
   /* Main Thread of the TcpNet. This task should have */
   /* the lowest priority because it is always READY. */
   dhcp_tout = DHCP_TOUT;
   while (1) {
      main_TcpNet();
      dhcp_check ();
      os_tsk_pass();
   }
}


int main (void) {
   /* Start with 'init' task. */
   os_sys_init(init);
   while(1);
}

#endif


/*----------------------------------------------------------------------------
 * end of file
 *---------------------------------------------------------------------------*/


