#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/poll.h>
#include <asm/irq.h>
#include <asm/io.h>
#include <linux/interrupt.h>
#include <asm/uaccess.h>
#include <asm/arch/regs-gpio.h>
#include <asm/hardware.h>
#include <asm/arch/regs-timer.h>
#include <asm/arch/regs-irq.h>
#include <asm/mach/time.h>
#include <asm/hardware/clock.h>

#define DEVICE_NAME  "qq2440_pwm"   /* ģʽִСcat /proc/devices豸 */
#define ADC_MAJOR    233         /* 豸 */

#define PWM_IOCTL_SET_FREQ		1
#define PWM_IOCTL_STOP			2

static struct semaphore lock;

/* freq:  pclk/50/16/65536 ~ pclk/50/16 
  * if pclk = 50MHz, freq is 1Hz to 62500Hz
  * human ear : 20Hz~ 20000Hz
  */
static void PWM_Set_Freq( unsigned long freq )
{
	unsigned long tcon;
	unsigned long tcnt;
	unsigned long tcfg1;
	unsigned long tcfg0;

	struct clk *clk_p;
	unsigned long pclk;

	//set GPB0 as tout0, pwm output
	s3c2410_gpio_cfgpin(S3C2410_GPB0, S3C2410_GPB0_TOUT0);

	tcon = __raw_readl(S3C2410_TCON);
	tcfg1 = __raw_readl(S3C2410_TCFG1);
	tcfg0 = __raw_readl(S3C2410_TCFG0);

	//prescaler = 50
	tcfg0 &= ~S3C2410_TCFG_PRESCALER0_MASK;
	tcfg0 |= (256 - 1); 

	//mux = 1/16
	tcfg1 &= ~S3C2410_TCFG1_MUX0_MASK;
	tcfg1 |= S3C2410_TCFG1_MUX0_DIV16;

	__raw_writel(tcfg1, S3C2410_TCFG1);
	__raw_writel(tcfg0, S3C2410_TCFG0);

	clk_p = clk_get(NULL, "pclk");
	pclk  = clk_get_rate(clk_p);
	tcnt  = (pclk/50/16)/freq;
	
	__raw_writel(tcnt, S3C2410_TCNTB(0));
	__raw_writel(tcnt/2, S3C2410_TCMPB(0));
				
	tcon &= ~0x1f;
	tcon |= 0xb;		//disable deadzone, auto-reload, inv-off, update TCNTB0&TCMPB0, start timer 0
	__raw_writel(tcon, S3C2410_TCON);
	
	tcon &= ~2;			//clear manual update bit
	__raw_writel(tcon, S3C2410_TCON);
}

void PWM_Stop( void )
{
	//set GPB0 as output
	s3c2410_gpio_cfgpin(S3C2410_GPB0, S3C2410_GPB0_OUTP);
	s3c2410_gpio_setpin(S3C2410_GPB0, 0);
}

/* Ӧó豸ļ/dev/adcִopen(...)ʱ
 * ͻs3c24xx_adc_open
 */
static int qq2440_pwm_open(struct inode *inode, struct file *file)
{
	if (!down_trylock(&lock))
		return 0;
	else
		return -EBUSY;
}


/* Ӧó豸ļ/dev/adcִclose(...)ʱ
 * ͻs3c24xx_adc_close
 */
static int qq2440_pwm_close(struct inode *inode, struct file *file)
{
	up(&lock);
    return 0;
}


/* Ӧó豸ļ/dev/adcִioctl(...)ʱ
 * ͻqq2440_pwm_ioctl
 */
static int qq2440_pwm_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
	switch (cmd) {
		case PWM_IOCTL_SET_FREQ:
			if (arg == 0)
				return -EINVAL;
			PWM_Set_Freq(arg);
			break;

		case PWM_IOCTL_STOP:
			PWM_Stop();
			break;
	}

	return 0;
}


/* ṹַ豸ĺ
 * Ӧó豸ļʱõopenreadwriteȺ
 * ջṹеĶӦ
 */
static struct file_operations qq2440_pwm_fops = {
    .owner   =   THIS_MODULE,    /* һָ꣬ģʱԶ__this_module */
    .open    =   qq2440_pwm_open,
    .release =   qq2440_pwm_close, 
    .ioctl   =   qq2440_pwm_ioctl,
};

static int __init qq2440_pwm_init(void)
{
    int ret;
	init_MUTEX(&lock);

    ret = register_chrdev(ADC_MAJOR, DEVICE_NAME, &qq2440_pwm_fops);
    if (ret < 0) {
      printk(DEVICE_NAME " can't register major number\n");
      return ret;
    }
	
#ifdef CONFIG_DEVFS_FS	
	devfs_mk_cdev(MKDEV(ADC_MAJOR, 0), S_IFCHR | S_IRUSR | S_IWUSR | S_IRGRP, DEVICE_NAME);
#endif

    printk(DEVICE_NAME " initialized\n");
    return 0;
}

/*
 * ִСrmmod s3c24xx_adc.koʱͻ 
 */
static void __exit qq2440_pwm_exit(void)
{
    /* ж */
#ifdef CONFIG_DEVFS_FS	
	devfs_remove(DEVICE_NAME);
#endif
    unregister_chrdev(ADC_MAJOR, DEVICE_NAME);
}

/* ָĳʼжغ */
module_init(qq2440_pwm_init);
module_exit(qq2440_pwm_exit);

/* һЩϢǱ */
MODULE_AUTHOR("http://www.arm9.net");             // 
MODULE_DESCRIPTION("S3C2410/S3C2440 PWM Driver");   // һЩϢ
MODULE_LICENSE("GPL");                              // ѭЭ

